<?php
/**
 * video-proxy.php — Stream video dari CDN dramaboxdb.com melalui server
 * Mengatasi CORS/hotlink protection di browser
 * Menggunakan cURL (lebih reliable dari fopen di XAMPP/Apache)
 *
 * Penggunaan: video-proxy.php?url=<encoded_video_url>
 */

// Whitelist domain CDN yang diizinkan
$allowedDomains = [
    'thwztvideo.dramaboxdb.com',
    'hwztvideo.dramaboxdb.com',
    'hwztakavideo.dramaboxdb.com',
    'thwztchapter.dramaboxdb.com',
    'hwztchapter.dramaboxdb.com',
    'hwzthls.dramaboxdb.com',
    'thwzthls.dramaboxdb.com',
    'dramaboxdb.com',
    'dramabox.apidrama.my.id',
    // Megawe CDN fallback
    'cdn.megawe.net',
    'megawe.net',
    'video.megawe.net',
    'media.megawe.net',
    // DramaBox additional CDN
    'dramabox.com',
    'thwzt.dramaboxdb.com',
    'hwzt.dramaboxdb.com',
    'cdn.dramaboxdb.com',
    'static.dramaboxdb.com',
];

$url = trim($_GET['url'] ?? '');
if (!$url) {
    http_response_code(400);
    exit('URL required');
}

// Decode jika URL-encoded
$url = urldecode($url);

// Validasi URL
if (!filter_var($url, FILTER_VALIDATE_URL)) {
    http_response_code(400);
    exit('Invalid URL');
}

$parsed = parse_url($url);
$host   = strtolower($parsed['host'] ?? '');

$allowed = false;
foreach ($allowedDomains as $d) {
    if ($host === $d || substr($host, -(strlen($d) + 1)) === '.' . $d) {
        $allowed = true;
        break;
    }
}
if (!$allowed) {
    http_response_code(403);
    exit('Domain not allowed');
}

// ── Disable output buffering ──
while (ob_get_level()) ob_end_clean();
set_time_limit(0);

// ── Curl request headers ──
$curlHeaders = [
    'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 Chrome/120 Safari/537.36',
    'Referer: https://www.dramabox.com/',
    'Accept: */*',
    'Accept-Encoding: identity',
];

// Support Range request (untuk video seek)
if (isset($_SERVER['HTTP_RANGE'])) {
    $curlHeaders[] = 'Range: ' . $_SERVER['HTTP_RANGE'];
}

// ── Headers sudah dikirim? flag ──
$headersSent = false;

// ── Curl header callback — tangkap dan forward response headers ──
$headerCallback = function ($ch, $headerLine) use (&$headersSent) {
    $len = strlen($headerLine);

    // HTTP status line
    if (preg_match('#^HTTP/\S+ (\d+)#i', $headerLine, $m)) {
        http_response_code((int)$m[1]);
        return $len;
    }

    $trimmed = trim($headerLine);
    if ($trimmed === '') {
        // End of headers — kirim CORS dan cache headers
        if (!$headersSent) {
            header('Access-Control-Allow-Origin: *');
            header('Access-Control-Allow-Methods: GET, HEAD, OPTIONS');
            header('Access-Control-Allow-Headers: Range');
            header('Access-Control-Expose-Headers: Content-Range, Accept-Ranges, Content-Length');
            header('Cache-Control: public, max-age=3600');
            header('X-Content-Type-Options: nosniff');
            $headersSent = true;
        }
        return $len;
    }

    // Forward relevant headers dari CDN ke browser
    if (preg_match('/^(Content-Type|Content-Length|Content-Range|Accept-Ranges|ETag|Last-Modified):\s*(.+)/i', $trimmed, $m)) {
        header($trimmed);
    }

    return $len;
};

// ── Curl write callback — stream body ke browser ──
$writeCallback = function ($ch, $data) {
    echo $data;
    flush();
    // Hentikan jika browser disconnect
    if (connection_aborted()) return 0;
    return strlen($data);
};

// ── Handle CORS preflight ──
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, HEAD, OPTIONS');
    header('Access-Control-Allow-Headers: Range');
    header('Access-Control-Max-Age: 86400');
    http_response_code(204);
    exit;
}

// ── Execute curl ──
$ch = curl_init();
curl_setopt_array($ch, [
    CURLOPT_URL            => $url,
    CURLOPT_RETURNTRANSFER => false,   // Jangan buffer di memory!
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_MAXREDIRS      => 5,
    CURLOPT_TIMEOUT        => 300,     // 5 menit untuk video besar
    CURLOPT_CONNECTTIMEOUT => 10,
    CURLOPT_SSL_VERIFYPEER => false,
    CURLOPT_SSL_VERIFYHOST => false,
    CURLOPT_HTTPHEADER     => $curlHeaders,
    CURLOPT_HEADERFUNCTION => $headerCallback,
    CURLOPT_WRITEFUNCTION  => $writeCallback,
    CURLOPT_BUFFERSIZE     => 256 * 1024,  // 256KB buffer
]);

// HEAD request jika browser minta HEAD
if ($_SERVER['REQUEST_METHOD'] === 'HEAD') {
    curl_setopt($ch, CURLOPT_NOBODY, true);
}

$result = curl_exec($ch);

if ($result === false && !$headersSent) {
    $error = curl_error($ch);
    http_response_code(502);
    header('Content-Type: text/plain');
    echo 'Proxy error: ' . $error;
}

curl_close($ch);
