<?php
/**
 * episode-fetch.php — AJAX endpoint untuk lazy-fetch video URL per episode
 *
 * Sumber (berurutan):
 *   1. DramaBox Web API → cover→video trick (semua episode termasuk premium)
 *   2. megawe.net /allepisode → video gratis (max ~26 ep)
 *
 * Request:  GET ?bookId=xxx&chapterId=yyy          (by chapterId)
 *           GET ?bookId=xxx&epIndex=5               (by episode index 0-based)
 * Response: JSON { "success": true, "mp4": "...", "m3u8": "...", "alt": [...] }
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

require_once __DIR__ . '/api.php';

$bookId    = preg_replace('/[^0-9]/', '', $_GET['bookId'] ?? '');
$chapterId = preg_replace('/[^0-9]/', '', $_GET['chapterId'] ?? '');
$epIndex   = isset($_GET['epIndex']) ? (int)$_GET['epIndex'] : -1;

if (!$bookId) {
    echo json_encode(['success' => false, 'error' => 'bookId diperlukan']);
    exit;
}
if (!$chapterId && $epIndex < 0) {
    echo json_encode(['success' => false, 'error' => 'chapterId atau epIndex diperlukan']);
    exit;
}

$mp4  = '';
$m3u8 = '';
$alt  = [];

// ── Sumber 1: DramaBox Web API via cover→video trick ──
$slug = discoverSlug($bookId);
if ($slug) {
    $pageProps = dramaboxDataRequest("/in/movie/$bookId/$slug.json");
    if (!$pageProps) {
        $pageProps = scrapeDramaboxPage("/in/movie/$bookId/$slug");
    }

    if ($pageProps) {
        $chapterList = $pageProps['chapterList'] ?? [];
        $foundChapter = null;

        foreach ($chapterList as $ch) {
            // Match by chapterId OR by index
            if ($chapterId && (string)($ch['id'] ?? '') === $chapterId) {
                $foundChapter = $ch;
                break;
            }
            if ($epIndex >= 0 && (int)($ch['index'] ?? -1) === $epIndex) {
                $foundChapter = $ch;
                // Set chapterId dari chapter yang ditemukan
                if (!$chapterId) $chapterId = (string)($ch['id'] ?? '');
                break;
            }
        }

        if ($foundChapter) {
            if (!empty($foundChapter['mp4']))     $mp4  = proxyUrl($foundChapter['mp4']);
            if (!empty($foundChapter['m3u8Url'])) $m3u8 = proxyUrl($foundChapter['m3u8Url']);

            $cover = $foundChapter['cover'] ?? '';
            if ($cover) {
                $coverUrls = coverToVideoUrls($cover);
                foreach ($coverUrls as $i => $cu) {
                    $proxied = proxyUrl($cu);
                    if ($i === 0 && !$mp4) {
                        $mp4 = $proxied;
                    } elseif ($proxied && $proxied !== $mp4) {
                        $alt[] = $proxied;
                    }
                }
            }
        }
    }
}

// ── Sumber 2: megawe /allepisode fallback ──
if (!$mp4 && !$m3u8) {
    $rEps = megaweRequest('/api/dramabox/allepisode', ['bookId' => $bookId, 'lang' => MEGAWE_LANG]);
    $megaweEps = $rEps['data'] ?? [];
    foreach ($megaweEps as $mep) {
        $matched = false;
        if ($chapterId && (string)($mep['chapterId'] ?? '') === $chapterId) $matched = true;
        if (!$matched && $epIndex >= 0 && (int)($mep['chapterIndex'] ?? -1) === $epIndex) $matched = true;
        if (!$matched) continue;

        $quals = extractAllQualitiesFromCdnList($mep['cdnList'] ?? []);
        if ($quals['mp4'])    $mp4  = proxyUrl($quals['mp4']);
        if ($quals['mp4_hd'] && $quals['mp4_hd'] !== $quals['mp4']) $alt[] = proxyUrl($quals['mp4_hd']);
        if ($quals['mp4_sd'] && $quals['mp4_sd'] !== $quals['mp4']) $alt[] = proxyUrl($quals['mp4_sd']);
        break;
    }
}

$alt = array_values(array_unique(array_filter($alt)));

echo json_encode([
    'success' => (bool)($mp4 || $m3u8),
    'mp4'     => $mp4,
    'm3u8'    => $m3u8,
    'alt'     => $alt,
]);
