<?php
/**
 * Boniflix API Helper v3.0
 * Sumber data utama: DramaBox Web API (dramaboxdb.com Next.js _next/data)
 * Sumber tambahan: megawe.net (search, dubbed, trending)
 * 
 * STRATEGI UNLOCK VIDEO:
 *   Cover image URL → video URL (replace .mp4.jpg@w=100&h=135 → .720p.narrowv3.mp4)
 *   Bekerja untuk SEMUA episode termasuk yang berbayar/premium!
 *
 * Fungsi API:
 *   getHome()                    → beranda dengan featured & sections
 *   getDetail($bookId)           → detail drama + SEMUA episode + video URLs
 *   getSearch($q, $page)         → cari drama
 *   getBrowse($page, $genreId)   → jelajahi drama
 *   getDubbed($classify, $page)  → drama dubbing Indonesia
 *   getLatest($page)             → drama terbaru
 *   getRanking($type)            → drama trending/ranking
 */

// ─── Konfigurasi ──────────────────────────────────────────────────────────────

define('DRAMABOX_DOMAIN', 'https://www.dramaboxdb.com');
define('MEGAWE_BASE',     'https://api.megawe.net');
define('MEGAWE_LANG',     'in');
define('CACHE_DIR',       __DIR__ . '/../cache/boniflix/');
define('CACHE_TTL',       600);           // 10 menit
define('CACHE_TTL_HOME',  300);           // 5 menit untuk home
define('CACHE_TTL_BUILD', 3600);          // 1 jam untuk build ID
define('CACHE_TTL_DETAIL', 300);          // 5 menit untuk detail drama (episode list)

// ─── Cache ────────────────────────────────────────────────────────────────────

function getCached($key) {
    if (!is_dir(CACHE_DIR)) return null;
    $file = CACHE_DIR . md5($key) . '.json';
    if (!file_exists($file)) return null;
    if ((time() - filemtime($file)) >= CACHE_TTL) return null;
    $d = @file_get_contents($file);
    return $d ? json_decode($d, true) : null;
}

function getCachedWithTTL($key, $ttl) {
    if (!is_dir(CACHE_DIR)) return null;
    $file = CACHE_DIR . md5($key) . '.json';
    if (!file_exists($file)) return null;
    if ((time() - filemtime($file)) >= $ttl) return null;
    $d = @file_get_contents($file);
    return $d ? json_decode($d, true) : null;
}

function setCache($key, $data) {
    if (!is_dir(CACHE_DIR)) @mkdir(CACHE_DIR, 0775, true);
    @file_put_contents(CACHE_DIR . md5($key) . '.json', json_encode($data));
}

// ─── HTTP Helpers ─────────────────────────────────────────────────────────────

/**
 * httpGet — HTTP GET request dengan retry dan error handling untuk shared hosting.
 * Banyak hosting shared punya masalah: DNS lambat, SSL strict, IP di-block, timeout kecil.
 * Fungsi ini melakukan retry sampai 2x dengan user-agent berbeda.
 */
function httpGet($url, $headers = [], $retries = 2) {
    // Rotasi User-Agent — beberapa CDN/WAF block UA tertentu
    $userAgents = [
        'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/131.0.0.0 Safari/537.36',
        'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/17.2 Safari/605.1.15',
        'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/130.0.0.0 Safari/537.36',
    ];

    for ($attempt = 0; $attempt <= $retries; $attempt++) {
        $ua = $userAgents[$attempt % count($userAgents)];
        $defaultHeaders = [
            'Accept: application/json, text/html, */*',
            'Accept-Language: en-US,en;q=0.9,id;q=0.8',
            'User-Agent: ' . $ua,
        ];
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 30,           // hosting lambat
            CURLOPT_CONNECTTIMEOUT => 15,            // hosting DNS lambat
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => 0,             // fix hosting yang CA cert outdated
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS      => 5,
            CURLOPT_ENCODING       => '',            // gzip/deflate
            CURLOPT_HTTPHEADER     => array_merge($defaultHeaders, $headers),
            CURLOPT_IPRESOLVE      => CURL_IPRESOLVE_V4, // force IPv4 — banyak hosting gagal IPv6
        ]);
        $resp = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $err  = curl_error($ch);
        curl_close($ch);

        if ($resp !== false && $code >= 200 && $code < 400) {
            return $resp;
        }

        // Jika 403/429/503 (rate-limit/block), tunggu sebentar sebelum retry
        if ($attempt < $retries) {
            usleep(500000 + $attempt * 500000); // 0.5s, 1s
        }
    }
    return null;
}

function megaweRequest($endpoint, $params = []) {
    if (!isset($params['lang'])) $params['lang'] = MEGAWE_LANG;

    $cacheKey = 'megawe_' . $endpoint . '?' . http_build_query($params);
    $cached = getCached($cacheKey);
    if ($cached !== null) return $cached;

    $url = MEGAWE_BASE . $endpoint;
    if (!empty($params)) $url .= '?' . http_build_query($params);

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => 30,
        CURLOPT_CONNECTTIMEOUT => 15,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_IPRESOLVE      => CURL_IPRESOLVE_V4,
        CURLOPT_HTTPHEADER     => ['Accept: application/json', 'User-Agent: Mozilla/5.0 (Linux; Android 14) AppleWebKit/537.36 Chrome/131.0 Safari/537.36'],
    ]);
    $resp = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($resp === false || $code >= 400) return null;
    $data = json_decode($resp, true);
    if (!$data) return null;
    if (!empty($data['success'])) setCache($cacheKey, $data);
    return $data;
}

// ─── DramaBox Web API ─────────────────────────────────────────────────────────

function getBuildId() {
    $cacheKey = 'dramabox_build_id_v2';
    $cached = getCachedWithTTL($cacheKey, CACHE_TTL_BUILD);
    if ($cached && !empty($cached['build_id'])) return $cached['build_id'];

    // Coba beberapa URL — kadang /in diblokir tapi path lain tidak
    $tryUrls = [
        DRAMABOX_DOMAIN . '/in',
        DRAMABOX_DOMAIN . '/',
        DRAMABOX_DOMAIN . '/en',
    ];

    foreach ($tryUrls as $tryUrl) {
        $html = httpGet($tryUrl);
        if ($html && preg_match('/"buildId"\s*:\s*"([^"]+)"/', $html, $m)) {
            $buildId = $m[1];
            setCache($cacheKey, ['build_id' => $buildId, 'ts' => time()]);
            return $buildId;
        }
    }

    return null;
}

function dramaboxDataRequest($path, $ttl = null) {
    $ttl = $ttl ?: CACHE_TTL;
    $cacheKey = 'dbox_data_' . $path;
    $cached = getCachedWithTTL($cacheKey, $ttl);
    if ($cached !== null) return $cached;

    $buildId = getBuildId();
    if (!$buildId) return null;

    $url = DRAMABOX_DOMAIN . "/_next/data/$buildId$path";
    $resp = httpGet($url);
    
    if (!$resp) {
        // Build ID mungkin sudah berubah — clear cache dan coba sekali lagi
        $file = CACHE_DIR . md5('dramabox_build_id_v2') . '.json';
        @unlink($file);
        $buildId = getBuildId();
        if (!$buildId) return null;
        $url = DRAMABOX_DOMAIN . "/_next/data/$buildId$path";
        $resp = httpGet($url);
        if (!$resp) return null;
    }

    $data = json_decode($resp, true);
    if (!$data || !isset($data['pageProps'])) return null;

    setCache($cacheKey, $data['pageProps']);
    return $data['pageProps'];
}

function scrapeDramaboxPage($path) {
    $cacheKey = 'dbox_scrape_' . $path;
    $cached = getCachedWithTTL($cacheKey, CACHE_TTL);
    if ($cached !== null) return $cached;

    $html = httpGet(DRAMABOX_DOMAIN . $path);
    if (!$html) return null;
    if (!preg_match('/<script id="__NEXT_DATA__"[^>]*>(.*?)<\/script>/s', $html, $m)) return null;

    $nextData = json_decode($m[1], true);
    if (!$nextData) return null;

    $pageProps = $nextData['props']['pageProps'] ?? [];
    if (!empty($pageProps)) setCache($cacheKey, $pageProps);
    return $pageProps;
}

// ─── Video URL dari Cover Image ───────────────────────────────────────────────

/**
 * Konversi cover image URL ke video URL.
 * Return array semua kemungkinan quality untuk fallback:
 *   [720p, 540p, 360p, 480p, original]
 */
function coverToVideoUrls($cover) {
    if (!$cover) return [];
    $qualities = ['720p', '480p', '540p', '360p'];
    $variants  = ['narrowv3', 'narrow', 'narrowv2'];
    $urls = [];
    foreach ($qualities as $q) {
        foreach ($variants as $v) {
            $suffix = ".$q.$v.mp4";
            // Pattern 1: .mp4.jpg@w=100&h=135
            $try = str_replace('.mp4.jpg@w=100&h=135', $suffix, $cover);
            if ($try !== $cover && !in_array($try, $urls)) { $urls[] = $try; continue 2; }
            // Pattern 2: .mp4.jpg@...
            $try = preg_replace('/\.mp4\.jpg(@[^"\'&\s]*)?/', $suffix, $cover);
            if ($try !== $cover && !in_array($try, $urls)) { $urls[] = $try; }
        }
    }
    return array_values(array_unique($urls));
}

function coverToVideoUrl($cover) {
    if (!$cover) return '';
    // Pattern 1: .mp4.jpg@w=100&h=135  → .720p.narrowv3.mp4
    $video = str_replace('.mp4.jpg@w=100&h=135', '.720p.narrowv3.mp4', $cover);
    if ($video !== $cover) return $video;
    // Pattern 2: .mp4.jpg@ (any query)
    $video = preg_replace('/\.mp4\.jpg(@[^"\'&\s]*)?/', '.720p.narrowv3.mp4', $cover);
    if ($video !== $cover) return $video;
    return '';
}

// ─── Normalisasi ──────────────────────────────────────────────────────────────

function normalizeDrama($d) {
    if (empty($d) || !is_array($d)) return null;
    $id = (string)($d['bookId'] ?? $d['book_id'] ?? '');
    if (!$id) return null;

    $tags = [];
    if (!empty($d['typeTwoNames']) && is_array($d['typeTwoNames'])) {
        $tags = $d['typeTwoNames'];
    } elseif (!empty($d['tags']) && is_array($d['tags'])) {
        foreach ($d['tags'] as $t) {
            if (is_string($t) && trim($t)) $tags[] = trim($t);
        }
    } elseif (!empty($d['tagV3s']) && is_array($d['tagV3s'])) {
        foreach ($d['tagV3s'] as $t) {
            if (!empty($t['tagName'])) $tags[] = $t['tagName'];
        }
    }
    if (empty($tags) && !empty($d['categoryName'])) $tags = [$d['categoryName']];

    return [
        'book_id'       => $id,
        'title'         => $d['bookName']      ?? $d['title']        ?? 'Untitled',
        'cover'         => $d['coverWap']      ?? $d['coverUrl']     ?? $d['cover'] ?? '',
        'introduction'  => $d['introduction']  ?? $d['description']  ?? '',
        'chapter_count' => (int)($d['chapterCount'] ?? $d['chapter_count'] ?? 0),
        'tags'          => $tags,
        'protagonist'   => $d['protagonist']   ?? $d['author']       ?? '',
        'rating'        => $d['rankVo']['hotCode'] ?? $d['viewCountDisplay'] ?? '',
        'completed'     => (bool)($d['completed'] ?? ($d['bookStatus'] ?? '') === 'END'),
        'genres'        => $tags,
        'slug'          => $d['bookNameLower'] ?? '',
    ];
}

function normalizeDramaList($list) {
    if (!is_array($list)) return [];
    $out = [];
    foreach ($list as $d) {
        $n = normalizeDrama($d);
        if ($n) $out[] = $n;
    }
    return $out;
}

// ─── Proxy URL ────────────────────────────────────────────────────────────────

function proxyUrl($rawUrl) {
    if (!$rawUrl) return '';
    if (strpos($rawUrl, '/') === 0 || strpos($rawUrl, 'video-proxy.php') !== false) return $rawUrl;
    // Auto-detect base path (localhost vs production)
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $basePath = (strpos($host, 'localhost') !== false || strpos($host, '127.0.0.1') !== false)
        ? '/dramabox-main/boniflix'
        : '';
    return $basePath . '/video-proxy.php?url=' . urlencode($rawUrl);
}

// ─── Slug helper ──────────────────────────────────────────────────────────────

function saveSlugMap($bookId, $slug) {
    if (!$bookId || !$slug) return;
    if (!is_dir(CACHE_DIR)) @mkdir(CACHE_DIR, 0775, true);
    $file = CACHE_DIR . 'slug_map.json';
    $map = [];
    if (file_exists($file)) $map = json_decode(@file_get_contents($file), true) ?: [];
    $map[$bookId] = $slug;
    @file_put_contents($file, json_encode($map));
}

function getSlug($bookId) {
    $file = CACHE_DIR . 'slug_map.json';
    if (!file_exists($file)) return null;
    $map = json_decode(@file_get_contents($file), true) ?: [];
    return $map[$bookId] ?? null;
}

function findSlugFromHome($bookId) {
    $slug = getSlug($bookId);
    if ($slug) return $slug;

    $homeData = dramaboxDataRequest('/in.json', CACHE_TTL_HOME);
    if (!$homeData) return null;

    foreach ($homeData['bigList'] ?? [] as $b) {
        saveSlugMap($b['bookId'] ?? '', $b['bookNameLower'] ?? '');
        if (($b['bookId'] ?? '') == $bookId) return $b['bookNameLower'] ?? null;
    }
    foreach ($homeData['smallData'] ?? [] as $sec) {
        foreach ($sec['items'] ?? [] as $b) {
            saveSlugMap($b['bookId'] ?? '', $b['bookNameLower'] ?? '');
            if (($b['bookId'] ?? '') == $bookId) return $b['bookNameLower'] ?? null;
        }
    }
    return getSlug($bookId);
}

function discoverSlug($bookId) {
    $slug = getSlug($bookId);
    if ($slug) return $slug;

    $slug = findSlugFromHome($bookId);
    if ($slug) return $slug;

    // Metode 2: Request /in/movie/{bookId} — redirect akan reveal slug
    $ch = curl_init(DRAMABOX_DOMAIN . '/in/movie/' . urlencode($bookId));
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => 30,
        CURLOPT_CONNECTTIMEOUT => 15,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_ENCODING       => '',
        CURLOPT_IPRESOLVE      => CURL_IPRESOLVE_V4,
        CURLOPT_HTTPHEADER     => ['User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/131.0.0.0 Safari/537.36'],
    ]);
    $html = curl_exec($ch);
    $effectiveUrl = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($httpCode == 200 && preg_match('/\/movie\/[^\/]+\/([a-z0-9\-]+)/i', $effectiveUrl, $m)) {
        $slug = $m[1];
        saveSlugMap($bookId, $slug);
        return $slug;
    }

    // Metode 3: __NEXT_DATA__ dari HTML redirect
    if ($html && preg_match('/<script id="__NEXT_DATA__"[^>]*>(.*?)<\/script>/s', $html, $m2)) {
        $nextData = json_decode($m2[1], true);
        $bi = $nextData['props']['pageProps']['bookInfo'] ?? [];
        $slug = $bi['bookNameLower'] ?? '';
        if ($slug) { saveSlugMap($bookId, $slug); return $slug; }
    }

    // Metode 4: Generate slug dari megawe detail title
    $megaweDetail = megaweRequest('/api/dramabox/detail', ['bookId' => $bookId, 'lang' => MEGAWE_LANG]);
    if ($megaweDetail && !empty($megaweDetail['data'])) {
        $d = $megaweDetail['data'];
        $title = $d['bookName'] ?? '';
        if ($title) {
            $genSlug = strtolower(preg_replace('/[^a-zA-Z0-9]+/', '-', trim($title)));
            $genSlug = trim($genSlug, '-');
            if ($genSlug) {
                // Verifikasi slug — cek apakah _next/data response valid
                $buildId = getBuildId();
                if ($buildId) {
                    $testUrl = DRAMABOX_DOMAIN . "/_next/data/$buildId/in/movie/$bookId/$genSlug.json";
                    $testResp = httpGet($testUrl);
                    if ($testResp) {
                        $testData = json_decode($testResp, true);
                        if (isset($testData['pageProps']['chapterList'])) {
                            saveSlugMap($bookId, $genSlug);
                            return $genSlug;
                        }
                    }
                }
            }
        }
    }

    return null;
}

// ─── Legacy compat (digunakan oleh episode-fetch.php) ─────────────────────────

function extractBestVideoFromCdnList($cdnList) {
    if (empty($cdnList) || !is_array($cdnList)) return '';
    $cdn = null;
    foreach ($cdnList as $c) { if (!empty($c['isDefault'])) { $cdn = $c; break; } }
    if (!$cdn) $cdn = $cdnList[0];
    $pathList = $cdn['videoPathList'] ?? [];
    if (empty($pathList)) return '';
    $free720 = $anyUrl = null;
    foreach ($pathList as $p) {
        $q = (int)($p['quality'] ?? 0); $url = $p['videoPath'] ?? ''; $vip = (bool)($p['isVipEquity'] ?? false);
        if (!$url) continue;
        if (!$anyUrl) $anyUrl = $url;
        if (!$vip && $q >= 720 && $q < 1080 && !$free720) $free720 = $url;
    }
    return $free720 ?: $anyUrl ?: '';
}

function extractAllQualitiesFromCdnList($cdnList) {
    $mp4 = ''; $mp4Hd = ''; $mp4Sd = '';
    if (empty($cdnList) || !is_array($cdnList)) return ['mp4' => '', 'mp4_hd' => '', 'mp4_sd' => ''];
    usort($cdnList, function($a, $b) { return ($b['isDefault'] ?? 0) - ($a['isDefault'] ?? 0); });
    foreach ($cdnList as $cdn) {
        foreach (($cdn['videoPathList'] ?? []) as $p) {
            $q = (int)($p['quality'] ?? 0); $url = $p['videoPath'] ?? '';
            if (!$url) continue;
            if ($q >= 1080 && !$mp4Hd) $mp4Hd = $url;
            if ($q >= 720 && $q < 1080 && !$mp4) $mp4 = $url;
            if ($q >= 540 && $q < 720 && !$mp4Sd) $mp4Sd = $url;
        }
    }
    if (!$mp4) $mp4 = $mp4Sd ?: $mp4Hd; if (!$mp4Hd) $mp4Hd = $mp4; if (!$mp4Sd) $mp4Sd = $mp4;
    return ['mp4' => $mp4, 'mp4_hd' => $mp4Hd, 'mp4_sd' => $mp4Sd];
}

// ─── API Functions ────────────────────────────────────────────────────────────

function getHome() {
    $homeData = dramaboxDataRequest('/in.json', CACHE_TTL_HOME);

    if ($homeData) {
        $bigList = $homeData['bigList'] ?? [];
        $smallData = $homeData['smallData'] ?? [];

        foreach ($bigList as $b) saveSlugMap($b['bookId'] ?? '', $b['bookNameLower'] ?? '');
        foreach ($smallData as $sec) {
            foreach ($sec['items'] ?? [] as $b) saveSlugMap($b['bookId'] ?? '', $b['bookNameLower'] ?? '');
        }

        $featured = normalizeDramaList($bigList);

        $sectionNames = [
            '精彩剧集' => '&#127909; Drama Menarik',
            '当前热播' => '&#128293; Trending Sekarang',
            '必看好剧' => '&#11088; Wajib Tonton',
        ];

        $sections = [];
        if (!empty($featured)) {
            $sections[] = ['name' => '&#127470;&#127465; Drama Pilihan', 'list' => $featured];
        }
        foreach ($smallData as $sec) {
            $rawName = $sec['name'] ?? '';
            $displayName = $sectionNames[$rawName] ?? $rawName;
            $list = normalizeDramaList($sec['items'] ?? []);
            if (!empty($list)) $sections[] = ['name' => $displayName, 'list' => $list];
        }

        // Tambahkan data megawe sebagai section ekstra
        $dubbed = megaweRequest('/api/dramabox/dubbed', ['classify' => 'popular', 'lang' => MEGAWE_LANG, 'page' => 1]);
        $dubbedList = normalizeDramaList($dubbed['data'] ?? []);
        if (!empty($dubbedList)) $sections[] = ['name' => '&#127470;&#127465; Dubbing Indonesia', 'list' => array_slice($dubbedList, 0, 20)];

        $trending = megaweRequest('/api/dramabox/trending', ['lang' => MEGAWE_LANG]);
        $trendingList = normalizeDramaList($trending['data'] ?? []);
        if (!empty($trendingList)) $sections[] = ['name' => '&#128200; Populer Minggu Ini', 'list' => array_slice($trendingList, 0, 20)];

        return ['success' => true, 'featured' => $featured, 'sections' => $sections];
    }

    // Fallback penuh ke megawe
    return getHomeFallback();
}

function getHomeFallback() {
    $dubbed   = megaweRequest('/api/dramabox/dubbed',   ['classify' => 'popular', 'lang' => MEGAWE_LANG, 'page' => 1]);
    $trending = megaweRequest('/api/dramabox/trending',  ['lang' => MEGAWE_LANG]);
    $latest   = megaweRequest('/api/dramabox/latest',    ['lang' => MEGAWE_LANG]);
    $foryou   = megaweRequest('/api/dramabox/foryou',    ['lang' => MEGAWE_LANG]);
    $dubbedList   = normalizeDramaList($dubbed['data']   ?? []);
    $trendingList = normalizeDramaList($trending['data'] ?? []);
    $latestList   = normalizeDramaList($latest['data']   ?? []);
    $foryouList   = normalizeDramaList($foryou['data']   ?? []);
    $featured = !empty($dubbedList) ? $dubbedList : $trendingList;
    $sections = [];
    if (!empty($dubbedList))   $sections[] = ['name' => '&#127470;&#127465; Dubbing Indonesia',   'list' => array_slice($dubbedList, 0, 20)];
    if (!empty($trendingList)) $sections[] = ['name' => '&#128293; Trending Sekarang',             'list' => array_slice($trendingList, 0, 20)];
    if (!empty($latestList))   $sections[] = ['name' => '&#127381; Baru Diupdate',                 'list' => array_slice($latestList, 0, 20)];
    if (!empty($foryouList))   $sections[] = ['name' => '&#127775; Rekomendasi Untukmu',           'list' => array_slice($foryouList, 0, 20)];
    return ['success' => true, 'featured' => $featured, 'sections' => $sections];
}

function getDubbed($classify = 'popular', $page = 1) {
    $p = max(1, (int)$page);
    $r = megaweRequest('/api/dramabox/dubbed', ['classify' => $classify, 'lang' => MEGAWE_LANG, 'page' => $p]);
    $list = normalizeDramaList($r['data'] ?? []);
    return ['success' => true, 'page' => $p, 'classify' => $classify, 'data' => $list, 'has_more' => count($list) >= 10];
}

function getBrowse($page = 1, $genre_id = null) {
    $p = max(1, (int)$page);

    if ($p == 1) {
        $homeData = dramaboxDataRequest('/in.json', CACHE_TTL_HOME);
        if ($homeData) {
            $allDramas = []; $seen = [];
            foreach ($homeData['bigList'] ?? [] as $b) {
                $bid = $b['bookId'] ?? '';
                if (!$bid || isset($seen[$bid])) continue;
                $seen[$bid] = true; saveSlugMap($bid, $b['bookNameLower'] ?? '');
                $n = normalizeDrama($b); if ($n) $allDramas[] = $n;
            }
            foreach ($homeData['smallData'] ?? [] as $sec) {
                foreach ($sec['items'] ?? [] as $b) {
                    $bid = $b['bookId'] ?? '';
                    if (!$bid || isset($seen[$bid])) continue;
                    $seen[$bid] = true; saveSlugMap($bid, $b['bookNameLower'] ?? '');
                    $n = normalizeDrama($b); if ($n) $allDramas[] = $n;
                }
            }
            if (!empty($allDramas)) {
                $r = megaweRequest('/api/dramabox/dubbed', ['classify' => 'latest', 'lang' => MEGAWE_LANG, 'page' => $p]);
                foreach (normalizeDramaList($r['data'] ?? []) as $m) {
                    if (!isset($seen[$m['book_id']])) { $seen[$m['book_id']] = true; $allDramas[] = $m; }
                }
                return ['success' => true, 'page' => $p, 'total_pages' => 50, 'genres' => [], 'data' => $allDramas];
            }
        }
    }

    $r = megaweRequest('/api/dramabox/dubbed', ['classify' => 'latest', 'lang' => MEGAWE_LANG, 'page' => $p]);
    $list = normalizeDramaList($r['data'] ?? []);
    if (empty($list)) { $r2 = megaweRequest('/api/dramabox/foryou', ['lang' => MEGAWE_LANG]); $list = normalizeDramaList($r2['data'] ?? []); }
    return ['success' => true, 'page' => $p, 'total_pages' => 50, 'genres' => [], 'data' => $list];
}

function getSearch($q, $page = 1) {
    $r = megaweRequest('/api/dramabox/search', ['query' => $q, 'lang' => MEGAWE_LANG]);
    $list = normalizeDramaList($r['data'] ?? []);
    return ['success' => true, 'total' => count($list), 'total_pages' => 1, 'data' => $list];
}

function getLatest($page = 1) {
    $r = megaweRequest('/api/dramabox/latest', ['lang' => MEGAWE_LANG]);
    $list = normalizeDramaList($r['data'] ?? []);
    return ['success' => true, 'page' => $page, 'total_pages' => 1, 'data' => $list];
}

function getRanking($type = 0) {
    $r = megaweRequest('/api/dramabox/trending', ['lang' => MEGAWE_LANG]);
    $raw = $r['data'] ?? [];
    return ['success' => true, 'type' => $type, 'data' => normalizeDramaList(is_array($raw) ? $raw : [])];
}

/**
 * Detail drama + SEMUA episode dengan video URLs (UNLOCK ALL).
 *
 * Strategi (3 sumber, berurutan):
 *   1. DramaBox Web API (_next/data) → bookInfo + chapterList LENGKAP (semua episode)
 *   2. Scrape HTML DramaBox → fallback jika _next/data gagal
 *   3. Megawe API → detail (info drama + chapterCount) + allepisode (video 21-26 ep)
 *
 * Jika hanya megawe tersedia, tetap generate SEMUA episode berdasarkan chapterCount.
 * Episode yang tidak punya video akan di-lazy-fetch via episode-fetch.php (client-side).
 */
function getDetail($bookId) {
    $bookId = preg_replace('/[^0-9]/', '', $bookId);
    if (!$bookId) return null;

    $cacheKey = 'detail_v4_' . $bookId;
    $cached = getCachedWithTTL($cacheKey, CACHE_TTL_DETAIL);
    if ($cached !== null) return $cached;

    $bookInfo = null;
    $chapterList = [];
    $slug = null;

    // ── Sumber 1: DramaBox Web API ──
    $slug = discoverSlug($bookId);

    if ($slug) {
        $pageProps = dramaboxDataRequest("/in/movie/$bookId/$slug.json");
        if ($pageProps) {
            $bookInfo    = $pageProps['bookInfo']    ?? null;
            $chapterList = $pageProps['chapterList'] ?? [];
        }
    }

    // ── Sumber 2: Scrape fallback ──
    if (empty($chapterList) && $slug) {
        $pageProps = scrapeDramaboxPage("/in/movie/$bookId/$slug");
        if ($pageProps) {
            $bookInfo    = $bookInfo ?: ($pageProps['bookInfo'] ?? null);
            $chapterList = $pageProps['chapterList'] ?? [];
        }
    }

    // ── Sumber 3: Megawe (SELALU panggil — untuk detail info & fallback video) ──
    $megaweDetail = null;
    $megaweEpsMap = [];  // indexed by chapterIndex

    $rDetail = megaweRequest('/api/dramabox/detail', ['bookId' => $bookId, 'lang' => MEGAWE_LANG]);
    if ($rDetail && !empty($rDetail['success'])) $megaweDetail = $rDetail['data'] ?? null;

    $rEps = megaweRequest('/api/dramabox/allepisode', ['bookId' => $bookId, 'lang' => MEGAWE_LANG]);
    $megaweRawEps = $rEps['data'] ?? [];
    if (is_array($megaweRawEps)) {
        foreach ($megaweRawEps as $mep) {
            $idx = (int)($mep['chapterIndex'] ?? -1);
            if ($idx >= 0) $megaweEpsMap[$idx] = $mep;
        }
    }

    // ── Build drama info ──
    $drama = null;
    if ($bookInfo) $drama = normalizeDrama($bookInfo);
    elseif ($megaweDetail) $drama = normalizeDrama($megaweDetail);

    if (!$drama && empty($chapterList) && empty($megaweEpsMap)) return null;

    // ── Tentukan total episode sebenarnya ──
    $apiChapterCount = 0;
    if ($bookInfo)     $apiChapterCount = max($apiChapterCount, (int)($bookInfo['chapterCount'] ?? 0));
    if ($megaweDetail) $apiChapterCount = max($apiChapterCount, (int)($megaweDetail['chapterCount'] ?? 0));
    if ($drama)        $apiChapterCount = max($apiChapterCount, (int)($drama['chapter_count'] ?? 0));
    $apiChapterCount = max($apiChapterCount, count($chapterList), count($megaweEpsMap));

    // ── Build episodes ──
    $episodes = [];

    if (!empty($chapterList)) {
        // Sumber 1/2 berhasil → gunakan chapterList (sudah lengkap)
        foreach ($chapterList as $ch) {
            $chId    = (string)($ch['id']    ?? '');
            $epIdx   = (int)($ch['index']    ?? 0);
            $epNum   = $epIdx + 1;
            $cover   = $ch['cover']          ?? '';
            $dur     = (int)($ch['duration'] ?? 0);

            // Dapatkan semua kemungkinan video URL dari cover (multi-quality)
            $coverVideoUrls = coverToVideoUrls($cover);
            $videoUrl = $coverVideoUrls[0] ?? coverToVideoUrl($cover);

            // Cek apakah megawe punya video untuk episode ini (lebih reliable kadang)
            $megaweMp4 = '';
            if (isset($megaweEpsMap[$epIdx])) {
                $mQuals = extractAllQualitiesFromCdnList($megaweEpsMap[$epIdx]['cdnList'] ?? []);
                $megaweMp4 = $mQuals['mp4'] ?: '';
            }

            // Prioritas: megawe video > field mp4 > cover-trick
            $mp4  = $megaweMp4 ?: (!empty($ch['mp4']) ? $ch['mp4'] : $videoUrl);
            $m3u8 = !empty($ch['m3u8Url']) ? $ch['m3u8Url'] : '';

            $proxiedMp4  = $mp4  ? proxyUrl($mp4)  : '';
            $proxiedM3u8 = $m3u8 ? proxyUrl($m3u8) : '';
            $hasVideo    = !empty($mp4) || !empty($m3u8);

            // Alt URLs untuk fallback audio
            $altUrls = [];
            if ($megaweMp4 && $videoUrl && $megaweMp4 !== $videoUrl) {
                $altUrls[] = proxyUrl($videoUrl); // cover-trick sebagai alt
            }
            foreach ($coverVideoUrls as $altU) {
                $p = proxyUrl($altU);
                if ($p && $p !== $proxiedMp4 && !in_array($p, $altUrls)) $altUrls[] = $p;
            }

            $episodes[] = [
                'chapter_id'     => $chId,
                'episode_number' => $epNum,
                'episode_name'   => $ch['name'] ?? ('Episode ' . $epNum),
                'cover'          => $cover,
                'duration_sec'   => $dur > 0 ? round($dur / 1000) : 0,
                'is_free'        => $hasVideo,
                'unlock'         => $hasVideo,
                'video'          => [
                    'mp4'     => $proxiedMp4,
                    'mp4_hd'  => $proxiedMp4,
                    'mp4_fhd' => $proxiedMp4,
                    'mp4_sd'  => $proxiedMp4,
                    'm3u8'    => $proxiedM3u8,
                    'alt'     => $altUrls,
                ],
            ];
        }
    } else {
        // Sumber 1/2 GAGAL → gunakan megawe + generate sisanya
        // Step A: Episode yang ada di megawe (punya video)
        foreach ($megaweEpsMap as $epIdx => $mep) {
            $epNum = $epIdx + 1;
            $quals = extractAllQualitiesFromCdnList($mep['cdnList'] ?? []);
            $episodes[] = [
                'chapter_id'     => (string)($mep['chapterId'] ?? $epIdx),
                'episode_number' => $epNum,
                'episode_name'   => $mep['chapterName'] ?? ('Episode ' . $epNum),
                'cover'          => '',
                'duration_sec'   => 0,
                'is_free'        => true,
                'unlock'         => true,
                'video'          => [
                    'mp4'     => $quals['mp4']    ? proxyUrl($quals['mp4'])    : '',
                    'mp4_hd'  => $quals['mp4_hd'] ? proxyUrl($quals['mp4_hd']) : '',
                    'mp4_fhd' => $quals['mp4_hd'] ? proxyUrl($quals['mp4_hd']) : '',
                    'mp4_sd'  => $quals['mp4_sd'] ? proxyUrl($quals['mp4_sd']) : '',
                    'm3u8'    => '',
                    'alt'     => [],
                ],
            ];
        }

        // Step B: Generate episode yang TIDAK ada di megawe (episode 27-63 dll)
        // Video-nya akan di-fetch on-demand via JS → episode-fetch.php
        for ($i = 0; $i < $apiChapterCount; $i++) {
            if (isset($megaweEpsMap[$i])) continue;  // sudah ada dari Step A
            $epNum = $i + 1;
            $episodes[] = [
                'chapter_id'     => '',  // unknown — akan di-resolve oleh episode-fetch.php
                'episode_number' => $epNum,
                'episode_name'   => 'Episode ' . $epNum,
                'cover'          => '',
                'duration_sec'   => 0,
                'is_free'        => true,
                'unlock'         => false,  // belum di-unlock, perlu fetch
                'video'          => [
                    'mp4'     => '',
                    'mp4_hd'  => '',
                    'mp4_fhd' => '',
                    'mp4_sd'  => '',
                    'm3u8'    => '',
                    'alt'     => [],
                ],
            ];
        }
    }

    usort($episodes, function($a, $b) { return $a['episode_number'] - $b['episode_number']; });

    $totalEps = max($apiChapterCount, count($episodes));

    if (!$drama) {
        $drama = ['book_id' => $bookId, 'title' => 'Drama ' . $bookId, 'cover' => '', 'introduction' => '', 'chapter_count' => $totalEps, 'tags' => [], 'protagonist' => '', 'rating' => '', 'completed' => false, 'genres' => [], 'slug' => $slug ?: ''];
    }
    $drama['chapter_count'] = $totalEps;

    $result = ['success' => true, 'book_id' => $bookId, 'detail' => $drama, 'episodes' => $episodes];
    setCache($cacheKey, $result);
    return $result;
}

function getRandomDrama() {
    $r = megaweRequest('/api/dramabox/randomdrama', ['lang' => MEGAWE_LANG]);
    return normalizeDramaList($r['data'] ?? []);
}

function getPopulerSearch() {
    $r = megaweRequest('/api/dramabox/populersearch', ['lang' => MEGAWE_LANG]);
    $raw = $r['data'] ?? [];
    if (!is_array($raw)) return [];
    $out = [];
    foreach ($raw as $item) {
        if (is_string($item) && trim($item)) {
            $out[] = trim($item);
        } elseif (is_array($item) && !empty($item['bookName'])) {
            $out[] = $item['bookName'];
        }
    }
    return array_slice($out, 0, 15);
}

// ─── Legacy compat: scrapeDramaboxChapters / scrapeDramaboxVideoUrl ───────────

function scrapeDramaboxChapters($bookId) {
    $slug = discoverSlug($bookId);
    if (!$slug) return null;
    $pageProps = dramaboxDataRequest("/in/movie/$bookId/$slug.json");
    if (!$pageProps) return null;
    return ['success' => true, 'book_info' => $pageProps['bookInfo'] ?? [], 'chapter_list' => $pageProps['chapterList'] ?? [], 'build_id' => getBuildId() ?: ''];
}

function scrapeDramaboxVideoUrl($bookId, $chapterId) {
    $chapters = scrapeDramaboxChapters($bookId);
    if (!$chapters || empty($chapters['chapter_list'])) return null;
    foreach ($chapters['chapter_list'] as $ch) {
        if ((string)($ch['id'] ?? '') === $chapterId) {
            $videoUrl = coverToVideoUrl($ch['cover'] ?? '');
            return ['success' => true, 'chapter_id' => $chapterId, 'free_source' => '', 'current' => ['mp4' => $videoUrl, 'm3u8Url' => $ch['m3u8Url'] ?? ''], 'chapter_list' => $chapters['chapter_list']];
        }
    }
    return null;
}
